% Center Pivot Irrigation Simulation with Uniformity
clc; clear; close all;

% Parameters
pivot_length = 607.7; % Length of pivot arm (m)
flow_rate = 0.01; % Flow rate per unit length (m³/s/m)
pivot_speed = 2; % Speed of pivot arm (m/min)
angular_increment =1; % Angle step for simulation (degrees)
efficiency_factor = 0.85; % Efficiency correction factor

% Uniformity settings
total_radius = pivot_length; % Maximum radius (m)
angular_steps = 0:360; % Angles to simulate (degrees)
sector_width = angular_increment * (pi / 180) * pivot_length; % Arc width (m)

% Initialize storage
applied_water = zeros(size(angular_steps)); % Applied water at each angle

% Simulation loop
for i = 1:length(angular_steps)
    % Angle in radians
    theta = angular_steps(i) * (pi / 180);
    
    % Calculate arc length for sector
    arc_length = theta * pivot_length; % Arc length (m)
    
    % Calculate time for pivot to pass through this angle
    time = arc_length / (pivot_speed / 60); % Time (s)
    
    % Calculate water applied
    applied_water(i) = flow_rate * time * efficiency_factor / sector_width; % m³/m²
end

% Uniformity calculations
avg_water = mean(applied_water); % Average water applied
CU = 100 * (1 - (sum(abs(applied_water - avg_water)) / (length(angular_steps) * mean(angular_steps)))); % Christiansen's Uniformity Coefficient

% Visualization
figure;
polarplot(deg2rad(angular_steps), applied_water, 'b-', 'LineWidth', 2);
title('Water Application Uniformity for Center Pivot');
rlim([0 max(applied_water)*1.1]);
legend(sprintf('CU = %.2f%%', CU), 'Location', 'Best');

% Additional plots
figure;
plot(angular_steps, applied_water, 'b-', 'LineWidth', 2);
hold on;
yline(avg_water, 'r--', 'LineWidth', 2);
xlabel('Angle (degrees)');
ylabel('Applied Water (m³/m²)');
title('Water Application Along Center Pivot Arm');
legend('Applied Water', 'Average Water');
grid on;

disp(['Christiansen''s Uniformity Coefficient (CU): ', num2str(CU), '%']);
